import { prisma } from '@/lib/prisma';

export interface Project {
    id: string;
    templateId: string;
    name: string;
    slug: string;
    data: Record<string, string>;
    createdAt: string;
    views?: number;
}

// Helper to map Prisma result to Project interface
function mapProject(p: any): Project {
    return {
        ...p,
        data: JSON.parse(p.data),
        createdAt: p.createdAt.toISOString(),
    };
}

export async function getProjects(userId?: string): Promise<Project[]> {
    const where = userId ? { userId } : {};
    const projects = await prisma.project.findMany({
        where,
        orderBy: { createdAt: 'desc' }
    });
    return projects.map(mapProject);
}

export async function saveProject(project: Project, userId?: string) {
    const dataString = JSON.stringify(project.data);

    const saved = await prisma.project.upsert({
        where: { id: project.id },
        update: {
            templateId: project.templateId,
            name: project.name,
            slug: project.slug,
            data: dataString,
            // If userId is provided, ensure it's linked (optional, maybe we don't want to change ownership)
            ...(userId ? { userId } : {}),
        },
        create: {
            id: project.id,
            templateId: project.templateId,
            name: project.name,
            slug: project.slug,
            data: dataString,
            createdAt: new Date(project.createdAt),
            views: 0,
            userId: userId,
        },
    });
    return mapProject(saved);
}

export async function getProjectBySlug(slug: string): Promise<Project | undefined> {
    const project = await prisma.project.findUnique({
        where: { slug },
    });
    return project ? mapProject(project) : undefined;
}

export async function incrementProjectViews(id: string) {
    try {
        const project = await prisma.project.update({
            where: { id },
            data: {
                views: { increment: 1 }
            }
        });
        return project.views;
    } catch (e) {
        return 0;
    }
}
