#!/bin/bash
# Backup Script for Landing Page Builder
# Creates backups of database and files

# Configuration
BACKUP_DIR="$HOME/backups"
APP_DIR="$HOME/app"
DATE=$(date +%Y%m%d_%H%M%S)
DB_NAME="your_database_name"
DB_USER="your_database_user"
DB_PASS="your_database_password"

# Create backup directory if it doesn't exist
mkdir -p "$BACKUP_DIR"

echo "🔄 Starting backup process..."
echo "📅 Date: $(date)"

# 1. Backup Database
echo "💾 Backing up database..."
mysqldump -u "$DB_USER" -p"$DB_PASS" "$DB_NAME" > "$BACKUP_DIR/db_backup_$DATE.sql"

if [ $? -eq 0 ]; then
    echo "✅ Database backup completed: db_backup_$DATE.sql"
    
    # Compress database backup
    gzip "$BACKUP_DIR/db_backup_$DATE.sql"
    echo "✅ Database backup compressed"
else
    echo "❌ Database backup failed!"
fi

# 2. Backup Application Files
echo "📦 Backing up application files..."
tar -czf "$BACKUP_DIR/app_backup_$DATE.tar.gz" \
    -C "$APP_DIR" \
    --exclude='node_modules' \
    --exclude='.next' \
    --exclude='logs' \
    --exclude='*.log' \
    .

if [ $? -eq 0 ]; then
    echo "✅ Application backup completed: app_backup_$DATE.tar.gz"
else
    echo "❌ Application backup failed!"
fi

# 3. Backup Environment Variables
echo "🔐 Backing up environment variables..."
if [ -f "$APP_DIR/.env" ]; then
    cp "$APP_DIR/.env" "$BACKUP_DIR/env_backup_$DATE"
    echo "✅ Environment variables backed up"
else
    echo "⚠️  .env file not found"
fi

# 4. Clean up old backups (keep last 7 days)
echo "🧹 Cleaning up old backups..."
find "$BACKUP_DIR" -name "db_backup_*.sql.gz" -mtime +7 -delete
find "$BACKUP_DIR" -name "app_backup_*.tar.gz" -mtime +7 -delete
find "$BACKUP_DIR" -name "env_backup_*" -mtime +7 -delete
echo "✅ Old backups cleaned"

# 5. Calculate total backup size
TOTAL_SIZE=$(du -sh "$BACKUP_DIR" | cut -f1)
echo "📊 Total backup size: $TOTAL_SIZE"

# 6. List recent backups
echo ""
echo "📋 Recent backups:"
ls -lh "$BACKUP_DIR" | grep "$DATE"

echo ""
echo "✅ Backup process completed successfully!"
echo "📁 Backup location: $BACKUP_DIR"

# Optional: Send backup to remote location
# Uncomment and configure if needed
# rsync -avz "$BACKUP_DIR/" user@remote-server:/backup/location/

exit 0

