import { NextResponse } from 'next/server';
import { prisma } from '@/lib/prisma';
import { auth } from '@/auth';
import { nanoid } from 'nanoid';
import { apiRateLimiter, getClientId } from '@/lib/rate-limit';

export async function GET(request: Request) {
    try {
        // Rate limiting
        const clientId = getClientId(request);
        const rateLimitResult = await apiRateLimiter.check(clientId);
        
        if (!rateLimitResult.success) {
            return NextResponse.json(
                { error: 'Too many requests. Please try again later.' },
                { 
                    status: 429,
                    headers: {
                        'X-RateLimit-Limit': '30',
                        'X-RateLimit-Remaining': '0',
                        'X-RateLimit-Reset': rateLimitResult.reset.toString(),
                    }
                }
            );
        }

        const session = await auth();

        if (!session?.user?.id) {
            return NextResponse.json({ error: 'Unauthorized' }, { status: 401 });
        }

        const projects = await prisma.project.findMany({
            where: {
                userId: session.user.id
            },
            orderBy: {
                createdAt: 'desc'
            }
        });

        // Parse data JSON string safely
        const parsedProjects = projects.map(p => ({
            ...p,
            data: (() => {
                try {
                    return JSON.parse(p.data);
                } catch {
                    return {};
                }
            })()
        }));

        return NextResponse.json(parsedProjects, {
            headers: {
                'X-RateLimit-Limit': '30',
                'X-RateLimit-Remaining': rateLimitResult.remaining.toString(),
                'X-RateLimit-Reset': rateLimitResult.reset.toString(),
            }
        });
    } catch (error) {
        console.error('Error in GET /api/projects:', error);
        return NextResponse.json({ error: 'Internal server error' }, { status: 500 });
    }
}

export async function POST(request: Request) {
    try {
        // Rate limiting
        const clientId = getClientId(request);
        const rateLimitResult = await apiRateLimiter.check(clientId);
        
        if (!rateLimitResult.success) {
            return NextResponse.json(
                { error: 'Too many requests. Please try again later.' },
                { status: 429 }
            );
        }

        const session = await auth();

        if (!session?.user?.id) {
            return NextResponse.json({ error: 'Unauthorized' }, { status: 401 });
        }

        const body = await request.json();

        // Validate required fields
        if (!body.templateId || !body.name || !body.data) {
            return NextResponse.json(
                { error: 'Missing required fields' },
                { status: 400 }
            );
        }

        // Validate data size (max 10MB)
        const dataStr = JSON.stringify(body.data);
        if (dataStr.length > 10 * 1024 * 1024) {
            return NextResponse.json(
                { error: 'Project data too large (max 10MB)' },
                { status: 413 }
            );
        }

        // Sanitize slug (alphanumeric and hyphens only)
        let slug = body.slug || nanoid(10);
        slug = slug.toLowerCase().replace(/[^a-z0-9-]/g, '-');

        const projectData = {
            id: body.id || nanoid(),
            templateId: body.templateId,
            name: body.name.substring(0, 255), // Limit name length
            slug: slug,
            data: dataStr,
            userId: session.user.id
        };

        const project = await prisma.project.upsert({
            where: { id: projectData.id },
            update: {
                templateId: projectData.templateId,
                name: projectData.name,
                slug: projectData.slug,
                data: projectData.data,
                userId: session.user.id // Ensure ownership
            },
            create: projectData
        });

        return NextResponse.json({
            ...project,
            data: JSON.parse(project.data)
        });
    } catch (error) {
        console.error('Error in POST /api/projects:', error);
        
        // Don't expose internal error details
        if (error instanceof Error && error.message.includes('Unique constraint')) {
            return NextResponse.json(
                { error: 'A project with this slug already exists' },
                { status: 409 }
            );
        }
        
        return NextResponse.json(
            { error: 'Failed to save project' },
            { status: 500 }
        );
    }
}
