// Rate Limiting for API routes
// Prevents abuse and DDoS attacks

interface RateLimitStore {
  [key: string]: {
    count: number;
    resetTime: number;
  };
}

const store: RateLimitStore = {};

// Clean up old entries every 10 minutes
setInterval(() => {
  const now = Date.now();
  Object.keys(store).forEach(key => {
    if (store[key].resetTime < now) {
      delete store[key];
    }
  });
}, 10 * 60 * 1000);

export interface RateLimitConfig {
  interval: number; // Time window in milliseconds
  uniqueTokenPerInterval: number; // Max number of unique tokens
  maxRequests: number; // Max requests per token
}

export class RateLimiter {
  private interval: number;
  private maxRequests: number;

  constructor(config: RateLimitConfig) {
    this.interval = config.interval;
    this.maxRequests = config.maxRequests;
  }

  async check(token: string): Promise<{ success: boolean; remaining: number; reset: number }> {
    const now = Date.now();
    const key = token;

    if (!store[key]) {
      store[key] = {
        count: 1,
        resetTime: now + this.interval,
      };
      return {
        success: true,
        remaining: this.maxRequests - 1,
        reset: store[key].resetTime,
      };
    }

    const tokenData = store[key];

    // Reset if time window has passed
    if (tokenData.resetTime < now) {
      tokenData.count = 1;
      tokenData.resetTime = now + this.interval;
      return {
        success: true,
        remaining: this.maxRequests - 1,
        reset: tokenData.resetTime,
      };
    }

    // Increment counter
    tokenData.count++;

    // Check if limit exceeded
    if (tokenData.count > this.maxRequests) {
      return {
        success: false,
        remaining: 0,
        reset: tokenData.resetTime,
      };
    }

    return {
      success: true,
      remaining: this.maxRequests - tokenData.count,
      reset: tokenData.resetTime,
    };
  }
}

// Default rate limiters
export const apiRateLimiter = new RateLimiter({
  interval: 60 * 1000, // 1 minute
  uniqueTokenPerInterval: 500,
  maxRequests: 30, // 30 requests per minute
});

export const authRateLimiter = new RateLimiter({
  interval: 15 * 60 * 1000, // 15 minutes
  uniqueTokenPerInterval: 500,
  maxRequests: 5, // 5 login attempts per 15 minutes
});

// Helper function to get client identifier
export function getClientId(request: Request): string {
  // Try to get IP from headers (useful when behind proxy/CDN)
  const forwarded = request.headers.get('x-forwarded-for');
  const realIp = request.headers.get('x-real-ip');
  const cfConnectingIp = request.headers.get('cf-connecting-ip');
  
  const ip = cfConnectingIp || realIp || forwarded?.split(',')[0] || 'unknown';
  
  return ip;
}

